<?php
require_once 'config.php';

// Set content type
header('Content-Type: application/json');

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    jsonResponse(false, 'Method not allowed', null, 405);
}

// Get JSON input
$jsonInput = file_get_contents('php://input');
$data = json_decode($jsonInput, true);

if (!$data) {
    jsonResponse(false, 'Invalid JSON data', null, 400);
}

// Sanitize input
$username = sanitizeInput($data['username'] ?? '');
$password = $data['password'] ?? '';

// Validate input
$validationRules = [
    'username' => ['required' => true, 'min' => 3, 'max' => 50],
    'password' => ['required' => true, 'min' => 6]
];

$errors = validateInput(['username' => $username, 'password' => $password], $validationRules);

if (!empty($errors)) {
    jsonResponse(false, 'Validation failed', $errors, 400);
}

// Verify credentials
if ($username !== ADMIN_USERNAME) {
    jsonResponse(false, 'Invalid username or password', null, 401);
}

if (!password_verify($password, ADMIN_PASSWORD_HASH)) {
    jsonResponse(false, 'Invalid username or password', null, 401);
}

// Create session
try {
    $sessionId = createSession($username);
    
    jsonResponse(true, 'Login successful', [
        'session_id' => $sessionId,
        'username' => $username,
        'expires_in' => JWT_EXPIRY
    ]);
    
} catch (Exception $e) {
    error_log('Login error: ' . $e->getMessage());
    jsonResponse(false, 'Login failed. Please try again.', null, 500);
}
?>